// Total number of frames
const frameCount = 65; // Adjust based on actual image count

// Function to get image URL from frame index
function getFrameUrl(index) {
  const frameNumber = String(index).padStart(3, "0");
  return `assets/frame_${frameNumber}.png`;
}

// Preload images
const images = [];
for (let i = 1; i <= frameCount; i++) {
  const img = new Image();
  img.src = getFrameUrl(i);
  images.push(img);
}

// Taglines
const taglines = [
  "Exhilarate every drive",
  "Precision in motion",
  "Engineered to perfection",
  "Unleash the power",
];

// References to elements
const carFrame = document.getElementById("car-frame");
const taglineEl = document.getElementById("tagline");

let currentTaglineIndex = -1;
let lastScrollY = window.scrollY;
let velocity = 0;
let isScrolling = false;
let targetFrame = 1;
let currentFrame = 1;
let smoothFactor = 0.1; // Adaptive interpolation factor
let stopTimeout = null; // To detect when scrolling completely stops

// Register GSAP plugin
gsap.registerPlugin(ScrollTrigger);

// Linear interpolation function for smooth transitions
function lerp(start, end, t) {
  return start * (1 - t) + end * t;
}

// Function to update the tagline
function updateTagline(progress) {
  const newTaglineIndex = Math.floor(progress * taglines.length);

  if (newTaglineIndex !== currentTaglineIndex && taglines[newTaglineIndex]) {
    currentTaglineIndex = newTaglineIndex;

    gsap.to(taglineEl, {
      duration: 0.3,
      scale: 0.5,
      opacity: 0,
      ease: "power2.in",
      onComplete: () => {
        taglineEl.textContent = taglines[newTaglineIndex];
        gsap.fromTo(
          taglineEl,
          { scale: 0.5, opacity: 0 },
          { duration: 0.6, scale: 1, opacity: 1, ease: "power2.out" }
        );
      },
    });
  }
}

// Function to smoothly animate frames
function animateFrames() {
  if (!isScrolling) {
    velocity *= 0.85; // Gradually reduce velocity
    if (Math.abs(velocity) < 0.01) velocity = 0; // Stop animation when very low
  }

  // Dynamically adjust interpolation smoothing
  smoothFactor = velocity > 1.5 ? 0.2 : velocity > 1.0 ? 0.15 : 0.1;

  // Smoothly interpolate between current and target frame
  currentFrame = lerp(currentFrame, targetFrame, smoothFactor);

  carFrame.src = getFrameUrl(Math.round(currentFrame));

  if (velocity !== 0 || Math.abs(targetFrame - currentFrame) > 0.5) {
    requestAnimationFrame(animateFrames);
  } else {
    // Ensure final frame sync when scrolling stops completely
    currentFrame = targetFrame;
    carFrame.src = getFrameUrl(Math.round(currentFrame));
  }
}

// Scroll event to track velocity
window.addEventListener("scroll", () => {
  const scrollY = window.scrollY;
  velocity = (scrollY - lastScrollY) * 0.08; // Adjust sensitivity
  lastScrollY = scrollY;
  isScrolling = true;

  if (velocity !== 0) {
    requestAnimationFrame(animateFrames);
  }

  // Detect complete stop (no scroll changes for 200ms)
  clearTimeout(stopTimeout);
  stopTimeout = setTimeout(() => {
    isScrolling = false;
    velocity = 0; // Force velocity to stop
    currentFrame = targetFrame; // Instantly snap to last frame
    carFrame.src = getFrameUrl(Math.round(currentFrame));
  }, 200);
});

// ScrollTrigger for syncing scroll position with animation
gsap.to(
  {},
  {
    scrollTrigger: {
      trigger: ".scroll-container",
      start: "top top",
      end: "bottom bottom",
      scrub: 1,
      onUpdate: (self) => {
        targetFrame = Math.min(
          frameCount,
          Math.max(1, Math.floor(self.progress * frameCount))
        );
        updateTagline(self.progress);
      },
    },
  }
);

// Initialize first frame and tagline
document.addEventListener("DOMContentLoaded", () => {
  carFrame.src = getFrameUrl(1);
  taglineEl.textContent = taglines[0];
  currentTaglineIndex = 0;
});
